<?php

namespace App\Http\Controllers\user;

use App\Http\Controllers\Controller;
use App\Http\Services\PaymentServices;
use App\Models\PaymentMethod;
use App\Models\User;
use App\Models\UserLedger;
use App\Models\Withdrawal;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Models\Deposit;
use App\Models\Purchase;
use App\Models\Setting;
use Carbon\Carbon;

class WithdrawController extends Controller
{
    public function withdraw_history()
{
    $withdraws = \App\Models\Withdrawal::where('user_id', auth()->id())
        ->orderByDesc('id')
        ->get();

    return view('app.main.withdraw_history', compact('withdraws'));
}

    public function withdraw()
    {
        if (auth()->user()->gateway_method && auth()->user()->gateway_address) {
            return view('app.main.withdraw.index');
        } else {
            return redirect()->route('user.bank')->with('success', 'First create a bank account');
        }
    }

    public function withdrawRequest(Request $request, PaymentServices $payment)
    {
        $now = Carbon::now('Africa/Harare'); // Adjust timezone if needed

        // ⏰ Only allow withdrawals from 10AM to 4PM
        if ($now->hour < 10 || $now->hour >= 16) {
            return redirect()->back()->with('error', 'Withdrawals are allowed only between 10:00 AM and 4:00 PM.');
        }

        // 🔁 Limit to 2 withdrawals per day
        $dailyWithdrawCount = Withdrawal::where('user_id', auth()->id())
            ->whereDate('created_at', Carbon::today())
            ->count();

        if ($dailyWithdrawCount >= 2) {
            return redirect()->back()->with('error', 'You can only withdraw 2 times per day.');
        }

        // ✅ Validation
        $validate = Validator::make($request->all(), [
            'amount' => 'required|numeric',
        ]);

        if ($validate->fails()) {
            return redirect()->back()->withErrors($validate)->withInput();
        }

        $user = auth()->user();
        $setting = Setting::first();

        // 💳 Check if user has deposited before
        $payments = Deposit::where('user_id', $user->id)->where('status', 'approved')->count();
        if ($payments < 1) {
            return redirect()->back()->with('error', "You can't withdraw before depositing.");
        }

        // 📦 Check investment
        if (!Purchase::where('user_id', $user->id)->exists()) {
            return redirect()->back()->with('error', "You need to invest before withdrawing.");
        }

        // ⚙️ Check withdraw availability
        if ($setting->open_transfer != 1) {
            return redirect()->back()->with('error', "Withdrawals are currently disabled. Try again later.");
        }

        // 💸 Validate amount
        $amount = $request->amount;
        if ($amount > $user->balance) {
            return redirect()->back()->with('error', 'Insufficient balance for withdrawal.');
        }

        if ($amount < setting('minimum_withdraw')) {
            return redirect()->back()->with('error', 'Minimum withdrawal is ' . setting('minimum_withdraw'));
        }

        if ($amount > setting('maximum_withdraw')) {
            return redirect()->back()->with('error', 'Maximum withdrawal is ' . setting('maximum_withdraw'));
        }

        $charge = 0;
        if (setting('withdraw_charge') > 0) {
            $charge = ($amount * setting('withdraw_charge')) / 100;
        }

        // Debit Wallet
        $debit_wallet = debit_user_wallet($user->id, 2, 'NGN', $amount);
        if ($debit_wallet['status'] == false) {
            return redirect()->back()->with('error', $debit_wallet['message']);
        }

        $finalAmount = $amount - $charge;
        $status_id = 'pending';
        $status_text = 'Withdraw request under review.';

        $reference = rand(10000,99999);

        if ($setting->auto_transfer) {
            $transfer = $payment->payout($reference, "NGN", $finalAmount, $setting->auto_transfer_default, $user->gateway_method, $user->gateway_address, $user->realname);

            if ($transfer['status'] == false) {
                credit_user_wallet($user->id, 2, 'NGN', $amount);
                return redirect()->back()->with('error', $transfer['message']);
            }

            $status_id = 'approved';
            $status_text = 'Your withdrawal has been approved.';
        }

        $paymenMethod = PaymentMethod::where('tag', $setting->auto_transfer_default)->first();

        $withdrawal = new Withdrawal();
        $withdrawal->user_id = $user->id;
        $withdrawal->method_name = $paymenMethod->name ?? '---';
        $withdrawal->trx = $reference;
        $withdrawal->account_info = json_encode([
            'bank_account' => $user->gateway_address,
            'full_name' => $user->realname,
            'bank_name' => $user->bank_name,
            'bank_code' => $user->gateway_method,
        ]);
        $withdrawal->number = $user->gateway_address;
        $withdrawal->amount = $amount;
        $withdrawal->currency = 'NGN';
        $withdrawal->charge = $charge;
        $withdrawal->oid = 'W-' . rand(100000,999999) . rand(100000,999999);
        $withdrawal->final_amount = $finalAmount;
        $withdrawal->status = $status_id;

        if ($withdrawal->save()) {
            $ledger = new UserLedger();
            $ledger->user_id = $user->id;
            $ledger->reason = 'withdraw_request';
            $ledger->perticulation = $user->bank_name . ' ' . $user->gateway_address;
            $ledger->amount = $amount;
            $ledger->debit = $finalAmount;
            $ledger->status = 'approved';
            $ledger->date = now()->format('d-m-Y H:i');
            $ledger->save();
        }

        return redirect()->back()->with('success', $status_text);
    }

    public function withdrawPreview()
    {
        $withdraws = Withdrawal::with('payment_method')
            ->where('user_id', Auth::id())
            ->orderByDesc('id')
            ->get();

        return view('app.main.withdraw_history', compact('withdraws'));
    }
}
